/*
 * Copyright (C) 2012 Edge-Core Networks
 * This software file (the "File") is owned and distributed by 
 * Edge-Core Networks under the following licensing terms.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */
#include <common.h>


#ifdef CONFIG_AMS_UBOOT
#include <ams_common.h>
//#include <ams_part.h>
#include <command.h>
#include "ams_ucmgr.h"

static unsigned long  uc_mem_start_addr;
static unsigned long  uc_mem_end_addr;
UC_MEM_Header_T       *uc_head = NULL;

/*
 * you can change UC_MGR_LOG_LEVEL to 4 or smaller to enable debug msg in uc_mgr.c
 * you should use AMS_PRINTK("<%d>...", UC_MGR_LOG_LEVEL, ...) 
 * if you are adding debug msg 
 */
#define UC_MGR_LOG_LEVEL 5

/* used to get physical memory size */
extern ulong get_effective_memsize(void);

static BOOL 
UC_MGR_IsValidDataPointer(void *ptr_addr, unsigned long size);


/*
 * FUNCTION NAME: UC_MGR_Reset
 * PURPOSE: This function is used to reset the un-cleared memory.
 * INPUT:   None.
 * OUTPUT:  None.
 * RETUEN:  None.
 * NOTES:   Clear the un-cleared memory include the header.
 *          After calling this function, the UC is invalid.
 */
void UC_MGR_Reset(void)
{
    unsigned long uc_start_addr;
    uc_start_addr = (unsigned long)(CONFIG_SYS_SDRAM_BASE + get_effective_memsize())
                        - CONFIG_AMS_UC_MEM_SIZE;

#if defined(CONFIG_MP) && (defined(CONFIG_MPC86xx) || defined(CONFIG_E500))
/* the dram top of 4K bytes will be used by uboot to setup second core,
 * UC memory is not able to use this 4K bytes, because the content will be changed
 * by loader in each reset. CONFIG_AMS_UC_MEM_SIZE will use CONFIG_PRAM to reserve
 * the UC memory region, this region will exclude the top 4K bytes.
 * refer to the function board_init_f() in lib_ppc/board.c.
 */
    uc_start_addr -= 0x1000; /* deduce 4K bytes to match the real start address reserved by CONFIG_PRAM */
#endif

    /* clear uc header */
    memset((char *)uc_start_addr, 0, CONFIG_AMS_UC_MEM_SIZE);
}


/*
 * FUNCTION NAME: UC_MGR_Init
 * PURPOSE: This function is used to initialize the un-cleared memory management module.
 * INPUT:   None.
 * OUTPUT:  None.
 * RETUEN:  TRUE    -- successful.
 *          FALSE   -- unspecified failure, system error.
 * NOTES:   Initialize the ucmgmt module.
 *          If the signature is correct, don't init the uc memory and use old one,
 *          this case may be warmboot, or watchdog reset. We need to keep the log information
 *          and store the log to logfile after reboot.
 */
#ifdef CONFIG_AMS_UCMGR_SHOWLOG
unsigned long g_dbg_buf;
#endif

BOOL UC_MGR_Init(void)
{
    UC_MGR_Sys_Info_T *p_sysinfo = 0;
    unsigned long     *p_reload_mode = 0;
    unsigned long     *p_reserved = 0;
#ifdef CONFIG_AMS_UCMGR_SHOWLOG
    UC_MGR_KERNEL_DBG_BUF_T *p_dbg_buf = 0;
#endif
    /* 
     * Loader can use physical address,
     * so get uc_mgr_uc_mem_start_addr directly.
     */
    uc_mem_start_addr = (unsigned long)(CONFIG_SYS_SDRAM_BASE + get_effective_memsize())
                        - CONFIG_AMS_UC_MEM_SIZE;
#if defined(CONFIG_MP) && (defined(CONFIG_MPC86xx) || defined(CONFIG_E500))
/* the dram top of 4K bytes will be used by uboot to setup second core,
 * UC memory is not able to use this 4K bytes, because the content will be changed
 * by loader in each reset. CONFIG_AMS_UC_MEM_SIZE will use CONFIG_PRAM to reserve
 * the UC memory region, this region will exclude the top 4K bytes.
 * refer to the function board_init_f() in lib_ppc/board.c.
 */
    uc_mem_start_addr -= 0x1000; /* deduce 4K bytes to match the real start address reserved by CONFIG_PRAM */
#endif

    uc_mem_end_addr = (unsigned long)(uc_mem_start_addr + CONFIG_AMS_UC_MEM_SIZE) - 1;

    uc_head = (UC_MEM_Header_T *)uc_mem_start_addr;
    AMS_ASSERT(uc_head == NULL, "uc_head = NULL\n");

    AMS_PRINTK("<%d>%s: UC_MEM_Header_T: %08x\n", UC_MGR_LOG_LEVEL, 
                      __FUNCTION__, (unsigned int)uc_head);

    if (memcmp(uc_head->signature, 
               CFG_AMS_UC_MGR_SIGNATURE, 
               CFG_AMS_UC_MGR_SIGNATURE_SIZE) == 0)
    {
#ifdef CONFIG_AMS_UCMGR_SHOWLOG
        g_dbg_buf = (unsigned long)(uc_head->offset[UC_MGR_DBG_BUF] + (unsigned char *)uc_head);
        flush_cache(g_dbg_buf, UC_MGR_KERNEL_DBG_BUF_SIZE);
#endif
        return TRUE;
    }

    /* clear uc memory */
    memset(uc_head, 0, CONFIG_AMS_UC_MEM_SIZE);

    /* init uc_head */
    memcpy(uc_head->signature, CFG_AMS_UC_MGR_SIGNATURE, CFG_AMS_UC_MGR_SIGNATURE_SIZE);
    uc_head->free_offset = L_CVRT_GET_OFFSET(uc_head, uc_head->data);

    /* allocate sysinfo block in uc memory, size is 1024 bytes */
    p_sysinfo = (UC_MGR_Sys_Info_T *)UC_MGR_Allocate(UC_MGR_SYS_INFO_INDEX, sizeof(UC_MGR_Sys_Info_T), sizeof(unsigned long));
    AMS_ASSERT(p_sysinfo == NULL, "failed to allocate sysinfo\n");

    /* allocate boot reason block in uc memory, size is 4 bytes */
    p_reload_mode = (unsigned long *)UC_MGR_Allocate(UC_MGR_BOOT_REASON_CHECKING_INDEX, sizeof(unsigned long), sizeof(unsigned long));
    AMS_ASSERT(p_reload_mode == NULL, "failed to allocate reload reason\n");

    /* allocate reserved block in uc memory, size is 1024 bytes */
    p_reserved = (unsigned long *)UC_MGR_Allocate(UC_MGR_RESERVED_INDEX, 1024, sizeof(unsigned long));
    AMS_ASSERT(p_reserved == NULL, "failed to allocate reserved block\n");

#ifdef CONFIG_AMS_UCMGR_SHOWLOG
    /* allocate reserved block in uc memory, size is  512k  bytes */
    p_dbg_buf = (UC_MGR_KERNEL_DBG_BUF_T*)UC_MGR_Allocate(UC_MGR_DBG_BUF, UC_MGR_KERNEL_DBG_BUF_SIZE, sizeof(unsigned long));
    AMS_ASSERT(p_dbg_buf == NULL, "failed to allocate p_dbg_buf block\n");

    g_dbg_buf = (unsigned long)p_dbg_buf;
    p_dbg_buf->start = 0;
    p_dbg_buf->end = 0;
    p_dbg_buf->size = UC_MGR_KERNEL_DBG_BUF_SIZE - 12;
    
    flush_cache(g_dbg_buf, UC_MGR_KERNEL_DBG_BUF_SIZE);
#endif
    return TRUE;
} /* End of UC_MGR_Init() */


/*
 * FUNCTION NAME: UC_MGR_Allocate
 * PURPOSE: This function is used to allocate memory from uc memory for need module.
 * INPUT:   alloc_index -- each data object must correspond a unique index value,
 *                         vaule range: 0 .. (SYS_ADPT_MAX_UC_BUFFER_POINT_INDEX_NUM-1).
 *          alloc_size  -- bytes to allocate.
 *          boundary    -- must order of 2
 * OUTPUT:  None.
 * RETUEN:  not 0   -- successful.
 *          0       -- unspecified failure, no free memory.
 * NOTES:   Must be called after UC_MGR_InitiateProcessResources.
 *
 */
void* UC_MGR_Allocate(unsigned long alloc_index, unsigned long alloc_size, unsigned long boundary)
{
    void *ptr;

    /* Check the alloc index */
    if (alloc_index > CFG_AMS_UC_MGR_INDEX_MAX)
        return NULL;

    /* Check the pointer address and end of data is excess the uc memory end address */
    if (uc_head->offset[alloc_index])
    {
        ptr = L_CVRT_GET_PTR(uc_head, uc_head->offset[alloc_index]);
        if (UC_MGR_IsValidDataPointer(ptr, alloc_size))
        {
            AMS_PRINTK("<%d>%s: alloc: %08x\n", UC_MGR_LOG_LEVEL, 
                       __FUNCTION__, (unsigned int)ptr);
            return ptr;
        }
    }

    /* has not allocated */
    ptr = (void*)L_ALIGN(L_CVRT_GET_PTR(uc_head, uc_head->free_offset), boundary);
    if (!UC_MGR_IsValidDataPointer(ptr, alloc_size))
    {
    	AMS_PRINTK("<%d>%s: alloc failed: %08x\n", UC_MGR_LOG_LEVEL, 
                       __FUNCTION__, (unsigned int)ptr);
        return NULL;
    }

    uc_head->offset[alloc_index] = L_CVRT_GET_OFFSET(uc_head, ptr);

    uc_head->free_offset = uc_head->offset[alloc_index] + alloc_size;

    AMS_PRINTK("<%d>%s: alloc: %08x\n", UC_MGR_LOG_LEVEL, 
               __FUNCTION__, (unsigned int)ptr);

    return ptr;
} /* End of UC_MGR_Allocate() */


/*
 * FUNCTION NAME: UC_MGR_GetSysInfo
 * PURPOSE: This function is used to get system information from uc memory.
 * INPUT:   *sys_info   -- output buffer of the system information.
 * OUTPUT:  *sys_info   -- the system information.
 * RETUEN:  TRUE    -- successful
 *          FALSE   -- failure
 * NOTES:   1. The function should be called by stktplg.
 *          2. The function should be called after the memory be allocated.
 *          3. If the memory is not allocated, the function will return FALSE.
 *
 */
BOOL UC_MGR_GetSysInfo(UC_MGR_Sys_Info_T *p_sysinfo)
{
    if (uc_head->offset[UC_MGR_SYS_INFO_INDEX] == 0)
        return FALSE; /* uc memory has not been allocated yet */

     *p_sysinfo = *(UC_MGR_Sys_Info_T *)L_CVRT_GET_PTR(uc_head, uc_head->offset[UC_MGR_SYS_INFO_INDEX]);
#if 0 /* for debug */
     set_verbose(1);
     printf("%s(%d):uc_sysinfo_p=0x%08lx\r\n", __FUNCTION__, __LINE__, (unsigned long)(L_CVRT_GET_PTR(uc_head, uc_head->offset[UC_MGR_SYS_INFO_INDEX])));
     printf("%s(%d):uc_sysinfo_p->raw.ext_info.burn_in_post_ever_failed=%d\r\n", __FUNCTION__, __LINE__, p_sysinfo->raw.ext_info.burn_in_post_ever_failed);
     printf("%s(%d):burn_in_post_ever_failed offset=%d\r\n", __FUNCTION__, __LINE__, (unsigned long)(&(p_sysinfo->raw.ext_info.burn_in_post_ever_failed)) - (unsigned long)p_sysinfo);
#endif

    return TRUE;
} /* End of UC_MGR_GetSysInfo() */

BOOL UC_MGR_GetReloadModeInfo(unsigned long *p_reload_mode)
{
    if (uc_head->offset[UC_MGR_BOOT_REASON_CHECKING_INDEX] == 0)
        return FALSE; /* uc memory has not been allocated yet */

    *p_reload_mode = *(unsigned long *)L_CVRT_GET_PTR(uc_head, uc_head->offset[UC_MGR_BOOT_REASON_CHECKING_INDEX]);

    return TRUE;
} 


BOOL UC_MGR_SetReloadModeInfo(unsigned long reload_mode)
{
    if (uc_head->offset[UC_MGR_BOOT_REASON_CHECKING_INDEX] == 0)
        return FALSE; /* uc memory has not been allocated yet */

    *(unsigned long *)L_CVRT_GET_PTR(uc_head, uc_head->offset[UC_MGR_BOOT_REASON_CHECKING_INDEX]) = reload_mode;

    return TRUE;
} /* End of UC_MGR_SetDownloadInfo() */

/*
 * FUNCTION NAME: UC_MGR_SetSysInfo
 * PURPOSE: This function is used to set system information to uc memory.
 * INPUT:   sys_info    -- setting value of the system information.
 * OUTPUT:  None.
 * RETUEN:  TRUE    -- successful
 *          FALSE   -- failure
 * NOTES:   1. The function should be called by prom code/load code.
 *          2. The function should be called after the memory be allocated.
 *          3. If the memory is not allocated, the function will return FALSE.
 *
 */
BOOL UC_MGR_SetSysInfo(UC_MGR_Sys_Info_T *p_sysinfo)
{
    int  i, size = sizeof(UC_MGR_Sys_Info_T);
    unsigned short *ptr = (unsigned short *)p_sysinfo;

    if (uc_head->offset[UC_MGR_SYS_INFO_INDEX] == 0)
        return FALSE; /* uc memory has not been allocated yet */

    p_sysinfo->check_sum = 0;
    for (i = 0; i < (size / 2) - 1; i++)
    {
        p_sysinfo->check_sum += ptr[i];
    }

    *(UC_MGR_Sys_Info_T *)L_CVRT_GET_PTR(uc_head, uc_head->offset[UC_MGR_SYS_INFO_INDEX]) = *p_sysinfo;
#if 0 /* for debug */
    set_verbose(1);
    printf("%s(%d):p_sysinfo->raw.ext_info.burn_in_post_ever_failed=%d\r\n", __FUNCTION__, __LINE__, p_sysinfo->raw.ext_info.burn_in_post_ever_failed);
#endif
    return TRUE;
} /* End of UC_MGR_SetSysInfo() */

#if 0 /* not used */
/*
 * FUNCTION NAME: UC_MGR_GetDownloadInfo
 * PURPOSE: This function is used to get download information from uc memory.
 * INPUT:   *download_info   -- output buffer of the download information.
 * OUTPUT:  *download_info   -- the download information.
 * RETUEN:  TRUE    -- successful
 *          FALSE   -- failure
 * NOTES:   1. The function should be called by download module.
 *          2. The function should be called after the memory be allocated.
 *          3. If the memory is not allocated, the function will return FALSE.
 *
 */
BOOL UC_MGR_GetDownloadInfo(UC_MGR_Download_Info_T *p_downloadinfo)
{
    if (uc_head->offset[UC_MGR_DOWNLOAD_INFO_INDEX] == 0)
        return FALSE; /* uc memory has not been allocated yet */

    *p_downloadinfo = *(UC_MGR_Download_Info_T *)L_CVRT_GET_PTR(uc_head, uc_head->offset[UC_MGR_DOWNLOAD_INFO_INDEX]);

    return TRUE;
} /* End of UC_MGR_GetDownloadInfo() */


/*
 * FUNCTION NAME: UC_MGR_SetDownloadInfo
 * PURPOSE: This function is used to set download information to uc memory.
 * INPUT:   download_info    -- setting value of the download information.
 * OUTPUT:  None.
 * RETUEN:  TRUE    -- successful
 *          FALSE   -- failure
 * NOTES:   1. The function should be called by prom code/load code.
 *          2. The function should be called after the memory be allocated.
 *          3. If the memory is not allocated, the function will return FALSE.
 *
 */
BOOL UC_MGR_SetDownloadInfo(UC_MGR_Download_Info_T *p_downloadinfo)
{
    if (uc_head->offset[UC_MGR_DOWNLOAD_INFO_INDEX] == 0)
        return FALSE; /* uc memory has not been allocated yet */

    *(UC_MGR_Download_Info_T *)L_CVRT_GET_PTR(uc_head, uc_head->offset[UC_MGR_DOWNLOAD_INFO_INDEX]) = *p_downloadinfo;

    return TRUE;
} /* End of UC_MGR_SetDownloadInfo() */

#ifdef SYS_CPNT_3COM_LOOPBACK_TEST 
/*
 * FUNCTION NAME: UC_MGR_GetLoopbackTestResult
 * PURPOSE:       This function is used to get loopback test result, that was
 *                set by DIAG, from UC memory.
 * INPUT:         test_result --- The loopback test result in the format of port bit map.
 *                                If loopback test fail, the bit is "1", otherwise "0".
 *                                The MSB of byte 0 is port 1,
 *                                the LSB of byte 0 is port 8,
 *                                the MSB of byte 1 is port 9,
 *                                the LSB of byte 1 is port 16,
 *                                ...
 *                                and so on.
 * OUTPUT:        None.
 * RETUEN:        TRUE        --- successful
 *                FALSE       --- failure
 * NOTES:         1. This API shall only be called by SWCTRL.
 *                2. The function should be called after the memory be allocated by using
 *                   index UC_MGR_LOOPBACK_TEST_RESULT_INDEX.
 *                3. If the memory is not allocated, the function will return FALSE.
 */
BOOL UC_MGR_GetLoopbackTestResult(unsigned char test_result[SYS_ADPT_NBR_OF_BYTE_FOR_1BIT_UPORT_LIST])
{
    if (uc_head->offset[UC_MGR_LOOPBACK_TEST_RESULT_INDEX] == 0)
        return FALSE; /* uc memory has not been allocated yet */

    memcpy(test_result,
           L_CVRT_GET_PTR(uc_head, uc_head->offset[UC_MGR_LOOPBACK_TEST_RESULT_INDEX]),
           SYS_ADPT_NBR_OF_BYTE_FOR_1BIT_UPORT_LIST);

    return TRUE;
}


/*
 * FUNCTION NAME: UC_MGR_SetLoopbackTestResult
 * PURPOSE:       This function is used to set loopback test result, that wiil be
 *                got by SWCTRL, to UC memory.
 * INPUT:         None.
 * OUTPUT:        test_result --- The loopback test result in the format of port bit map.
 *                                If loopback test fail, the bit is "1", otherwise "0".
 *                                The MSB of byte 0 is port 1,
 *                                the LSB of byte 0 is port 8,
 *                                the MSB of byte 1 is port 9,
 *                                the LSB of byte 1 is port 16,
 *                                ...
 *                                and so on.
 * RETUEN:        TRUE        --- successful
 *                FALSE       --- failure
 * NOTES:         1. This API shall only be called by DIAG.
 *                2. The function should be called after the memory be allocated by using
 *                   index UC_MGR_LOOPBACK_TEST_RESULT_INDEX.
 *                3. If the memory is not allocated, the function will return FALSE.
 */
BOOL UC_MGR_SetLoopbackTestResult(unsigned char test_result[SYS_ADPT_NBR_OF_BYTE_FOR_1BIT_UPORT_LIST])
{
    if (uc_head->offset[UC_MGR_LOOPBACK_TEST_RESULT_INDEX] == 0)
        return FALSE; /* uc memory has not been allocated yet */

     memcpy(L_CVRT_GET_PTR(uc_head, uc_head->offset[UC_MGR_LOOPBACK_TEST_RESULT_INDEX]),
           test_result,
           SYS_ADPT_NBR_OF_BYTE_FOR_1BIT_UPORT_LIST);

    return TRUE;
}
#endif
#endif

/*
 * FUNCTION NAME: UC_MGR_GetUcMemStartAddr
 * PURPOSE: This function is used to get UC memory start address.
 * INPUT:   none
 * OUTPUT:  none
 * RETUEN:  UC memory start address
 *
 * NOTES:   Before getting uc memory start address
 *          UC_MGR_Init should be called first!!
 */
unsigned long UC_MGR_GetUcMemStartAddr(void)
{
    return uc_mem_start_addr;
} /* End of UC_MGR_GetUcMemStartAddr */

/*
 * FUNCTION NAME: UC_MGR_GetUcMemEndAddr
 * PURPOSE: This function is used to get UC memory end address.
 * INPUT:   none
 * OUTPUT:  none
 * RETUEN:  UC memory end address
 *
 * NOTES:   Before getting uc memory end address
 *          UC_MGR_Init should be called first!!
 */
unsigned long UC_MGR_GetUcMemEndAddr(void)
{
    return uc_mem_end_addr;
} /* End of UC_MGR_GetUcMemEndAddr */

/*
 * FUNCTION NAME: UC_MGR_IsValidDataPointer
 * PURPOSE: This function is used to check the uc pointer valid or not.
 * INPUT:   ptr_addr    -- pointer address.
 *          size        -- buffer size.
 * OUTPUT:  None.
 * RETUEN:  TRUE    -- successful
 *          FALSE   -- failure
 * NOTES:
 */
static BOOL UC_MGR_IsValidDataPointer(void *ptr_addr, unsigned long size)
{
    if (((unsigned long)ptr_addr < uc_mem_start_addr) 
        || ((unsigned long)ptr_addr + size - 1) > uc_mem_end_addr)
        return FALSE;

    return TRUE;
} /* end of UC_MGR_IsValidDataPointer */
#endif

